/* MakeData - explore unexplored data
  Run as assigning symbols and running MakeCode
  2010-02-18 Baseline
  2010-02-27 Clean up
  2010-03-15 Rework to allow cut/paste into other .idc
  2010-03-16 Rework to be generic
  2011-08-31 Avoid infinite loop; correct missing %lx formatting
*/

#include <idc.idc>

/**
 * Scan one data segment for unexplored, labeled data and explore
 */

static makeLabeledData(n, startEa, endEa)
{
  auto s;
  auto ea;
  auto ea2;
  auto flags;
  auto b1;
  auto b;
  auto len;
  auto ok;
  auto numSame;
  auto dlrEa;
  auto nulEa;
  auto isAscii;
  auto sEndEa;

  sEndEa = " and " + form("0x%lx", endEa);

  s = "makeLabeledData exploring labeled data in segment " + n +
      " between " + form("0x%lx", startEa) + sEndEa;
  Message(s + "\n");			// Put in log
  Warning(s);			// Enable for debug

  ea = startEa;

  while (1) {
    // s = "Finding unexplored between ea " + form("0x%lx", ea) + sEndEa;	// Enable for debug
    // Warning(s);			// Enable for debug
    Message(s + "\n");		// Enable for debug

    // Find next unexplored item
    flags = GetFlags(ea);
    if (!isUnknown(flags)) {
      ea = FindUnexplored(ea, 1);
      if (ea == BADADDR)
	break;
    }
    if (ea > endEa)
      break;

    s = "Found unexplored ea " + form("0x%lx", ea);	// Enable for debug
    // Warning(s);			// Enable for debug
    Message(s + "\n");		// Enable for debug

    s = Name(ea);

    if (s == "")
      ea = ea + 1;
    else {
      // Got a label
      ea2 = ea;
      b1 = Byte(ea);
      numSame = 1;
      isAscii = (b1 >= 0x20 && b1 <= 0x7f) || b1 == 0xa || b1 == 0xd;
      dlrEa = BADADDR;
      nulEa = BADADDR;
      // Find end of string or array bounds
      while (1) {
	ea2++;
	if (ea2 > endEa) break;
	s = Name(ea2);
	if (s != "") break;		// Stop at next label
	b = Byte(ea2);
	if (b1 == b)
	  numSame++;
	if (b == '$') {
	  if (dlrEa == BADADDR)
	    dlrEa = ea2;
	}
	else if (b == 0) {
	  if (nulEa == BADADDR)
	    nulEa = ea2;
	}
	else if ((b < 0x20 || b > 0x7f) && b != 0xa && b != 0xd)
	  isAscii = 0;
      } // while

      len = ea2 - ea;

      s = Name(ea) +
	 " ea " + form("0x%lx", ea) +
	 " ea2 " + form("0x%lx", ea2) +
	 " ascii " + form("%d", isAscii) +
	 " nulEa " + form("0x%lx", nulEa) +
	 " dlrEa " + form("0x%lx", dlrEa) +
	 " len " + form("%d", len);
      // Warning(s);			// Enable for debug

      if (isAscii) {
	// Got ascii string or array
	if (dlrEa != BADADDR) {
	  ok = MakeArray(ea, len);	// Assume driver name or $ term string
	  OpChr(ea, -1);
	}
	else if (nulEa != BADADDR)
	  ok = MakeStr(ea, nulEa - ea + 1);
	else
	  ok = MakeStr(ea, len);	// Assume not nul terminated
	if (!ok) {
	  s = s + " makeLabeledData MakeStr failed";
	  Message(s + "\n");		// Put in log
	  Warning(s);			// Complain
	}
      }	// if ascii

      else {
	// Got some kind of data
	// Warning("got data at " + form("0x%lx", ea));
	if (len == 1) {
	  MakeByte(ea);
	}
	else if (len == 2) {
	  MakeWord(ea);
	}
	else if (len == 3) {
	  MakeArray(ea, 3);
	}
	else if (len == 4) {
	  MakeDword(ea);
	}
	else if (len > 4) {
	  ok = MakeArray(ea, len);
	  if (!ok) {
	    s = s + " makeLabeledData MakeArray failed";
	    Message(s + "\n");		// Put in log
	    Warning(s);			// Complain
	  }
	  OpDecimal(ea, 0);
	}
	else {
	  Warning("Help");		// Complain
	  return;
	}
      } // if data
    } // if labeled
  } // while unexplored

  s = "makeLabeledData finished";
  Message(s + "\n");			// Put in log
  // Warning(s);			// Enable for debug

} // makeLabeledData

/**
 * Scan one data segment for unexplored misc data and explore
 */

static makeMiscData(n, startEa, endEa)
{
  auto s;
  auto ea;
  auto ea2;
  auto flags;
  auto b1;
  auto b;
  auto len;
  auto ok;
  auto numSame;
  auto dlrEa;
  auto nulEa;
  auto isAscii;

  s = "makeMiscData exploring unlabeled data in segment " + n + " between " + form("0x%lx", startEa) +
      " and " + form("0x%lx", endEa);
  Message(s + "\n");			// Put in log
  Warning(s);			// Enable for debug

  ea = startEa;

  while (1) {
    s = "Finding unexplored ea " + form("0x%lx", ea);	// Enable for debug
    // Warning(s);			// Enable for debug
    Message(s + "\n");		// Enable for debug

    // Find next unexplored item
    flags = GetFlags(ea);
    if (!isUnknown(flags)) {
      ea = FindUnexplored(ea, 1);
      if (ea == BADADDR)
	break;
    }
    if (ea > endEa)
      break;

    s = "Found unexplored ea " + form("0x%lx", ea);	// Enable for debug
    // Warning(s);			// Enable for debug
    Message(s + "\n");		// Enable for debug

    ea2 = ea;
    b1 = Byte(ea);
    numSame = 1;
    isAscii = (b1 >= 0x20 && b1 <= 0x7f) || b1 == 0xa || b1 == 0xd;
    dlrEa = BADADDR;
    nulEa = BADADDR;

    while (1) {
      ea2++;
      flags = GetFlags(ea2);
      if (ea2 > endEa) break;
      if (!isUnknown(flags)) break;
      b = Byte(ea2);
      if (b1 == b)
	numSame++;
      if (b == '$') {
	if (dlrEa == BADADDR)
	  dlrEa = ea2;
      }
      else if (b == 0) {
	if (nulEa == BADADDR)
	  nulEa = ea2;
      }
      else if ((b < 0x20 || b > 0x7f) && b != 0xa && b != 0xd)
	isAscii = 0;
      else if (b1 == 0)
	break;				// Start of new string?
    } // while

    len = ea2 - ea;

    s = " ea " + form("0x%lx", ea) +
	" ea2 " + form("0x%lx", ea2) +
	" ascii " + form("%d", isAscii) +
	" nulEa " + form("0x%lx", nulEa) +
	" dlrEa " + form("0x%lx", dlrEa) +
	" len " + form("%d", len);
    // Warning(s);			// Enable for debug

    if (isAscii) {
      // Got ascii string
      if (dlrEa != BADADDR) {
	// $ terminated
	ok = MakeArray(ea, len);
	OpChr(ea, -1);
      }
      else if (nulEa != BADADDR)
	ok = MakeStr(ea, nulEa - ea + 1);	// Nul terminated
      else
	ok = MakeStr(ea, len);		// Assume not nul terminated
      if (!ok) {
	s = s + " makeMiscData MakeStr failed";
	Message(s + "\n");		// Put in log
	Warning(s);
      }
    }	// if ascii

    else {
      // Got some kind of data
      // Warning(s);
      if (len == 1) {
	MakeByte(ea);
      }
      else if (len == 2) {
	MakeWord(ea);
      }
      else if (len == 3) {
	MakeArray(ea, 3);
      }
      else if (len == 4) {
	MakeDword(ea);
      }
      else if (len > 4) {
	ok = MakeArray(ea, len);
	if (!ok) {
	  s = s + " MakeArray failed";
	  Message(s + "\n");		// Put in log
	  Warning(s);			// Complain
	}
	OpDecimal(ea, 0);
      }
      else {
	Warning("Help");
	return;
      }
    } // if data
  } // while

  s = "makeMiscData finished";
  Message(s + "\n");			// Put in log

} // makeMiscData

/**
 * Explore strategy table
 */

static makeStrategyTbl()
{
  auto s;
  auto ea;
  auto csegBaseEa;
  auto startEa;
  auto endEa;
  auto ok;

  // Find code segment
  ea = SegByName("cseg02");
  if (ea == BADADDR) {
    s = "Can not find cseg02 by name";
    Message(s + "\n");			// Put in log
    Warning(s);				// Complain
    return;
  }

  csegBaseEa = ea;

  // Find data segment
  ea = SegByName("dseg01");
  if (ea == BADADDR) {
    s = "Can not find dseg01 by name";
    Message(s + "\n");			// Put in log
    Warning(s);				// Complain
    return;
  }

  s = SegName(ea);
  if (s != "dseg01") {
    s = "Can not find dseg01 by ea";
    Message(s + "\n");			// Put in log
    Warning(s);				// Complain
    return;
  }

  ea = LocByName("_g_StratList");

  if (ea == BADADDR) {
    s = "Can not find _g_StratList by name";
    Message(s + "\n");			// Put in log
    Warning(s);				// Complain
    return;
  }

  endEa = LocByName("_Device_Help");

  s = "makeStrategyTbl building table between " + form("0x%lx", ea) +
      " and " + form("0x%lx", endEa);
  Message(s + "\n");			// Put in log

  startEa = ea;

  while (ea < endEa) {
    MakeWord(ea);
    OpOff(ea, 0, csegBaseEa);
    ea = ea + 2;
  } // while

  s = "makeStrategyTbl finished";
  Message(s + "\n");			// Put in log

} // makeStrategyTbl


/**
 * analyze all data segments
 */

static analyzeDataSegs()
{
  auto s;
  auto ea;
  auto n;
  auto endEa;

  s = "analyzeDataSegs starting";
  Message("\n" + s + "\n");		// Put in log

  ea = FirstSeg();

  while (ea != BADADDR) {

    n = SegName(ea);
    if (n == "") {
      s = "Can not find name for segment at " + form("0x%lx", ea);
      Message(s + "\n");		// Put in log
      Warning(s);			// Complain
      return;
    }

    endEa = SegEnd(ea) - 1;

    // Assume data segs named dseg???
    if (substr(n, 0, 4) == "dseg") {
      makeLabeledData(n, ea, endEa);
      makeMiscData(n, ea, endEa);
    }

    ea = NextSeg(ea);

  } // while

  s = "analyzeDataSegs finished";
  Message(s + "\n\n");			// Put in log
  // Warning(s);			// Enable for debug

} // analyzeDataSegs

/**
 * main
 */

static main()
{
  auto s;

  s = "MakeData starting";
  Message("\n" + s + "\n");		// Put in log

  // makeStrategyTbl();

  analyzeDataSegs();

  s = "MakeData finished";
  Message(s + "\n\n");			// Put in log

} // main
