/* idashell - Open ida shell in current or selected directory
   arg is directory override

   Copyright (c) 2011, 2012 Steven Levine and Associates, Inc.
   All rights reserved.

   2010-12-19 SHL Baseline - clone from yah2stk
   2011-02-15 SHL Allow 4 open instances without WPS close request
   2011-03-03 SHL Better title
   2011-06-07 SHL Yet more better title
   2011-06-11 SHL Correct objectid naming
   2011-09-13 SHL Drop unused
   2011-10-04 SHL Try again for better title
   2012-07-23 SHL Add missing SETs for DOS settings
*/

signal on ERROR
signal on FAILURE name Error
signal on HALT
signal on NOTREADY name Error
signal on NOVALUE name Error
signal on SYNTAX name Error

if address() \= 'CMD' then do
  say 'Must run from command line'
  exit
end

call Initialize

G.!Version = '0.1'

Main:

  /* arg is directory name override */
  parse arg cmdLine
  call ScanArgs2 cmdLine
  drop CmdLine

  seqFile = MakePath(G.!TmpDir, G.!CmdName, '.seq')

  if IsFile(seqFile) then do
    objectSeq = linein(seqFile)
    call stream seqFile, 'C', 'CLOSE'
    if objectSeq >= 3 then
      objectSeq = 0
    else
      objectSeq = objectSeq + 1
  end
  else
    objectSeq = 0

  /* Build ida object and run it
     Use 4dos standard cmd sep
   */

  dirName = G.!DirName

  if dirName \== '' then do
    s = directory(dirName)
    if s == '' then call Fatal 'Can not switch to' dirName 'directory'
  end
  else
    dirName = directory()

  iSlash = lastpos('\', dirName)
  if iSlash == 0 then
    s = dirName
  else
    s = substr(dirName, iSlash + 1)
  /* Include parent in title unless well-known */
  if iSlash > 1 then do
    j = lastpos('\', dirName, iSlash - 1)
    if substr(dirName, j + 1, 8) \== 'ida_work' then
      s = substr(dirName, j + 1)
  end

  title = 'IDA shell for' s

  locId = '<WP_NOWHERE>'

  parms = '/K idax'
  objectId = 'IdaShell'
  if objectSeq \= 0 then
    objectId = objectId || objectSeq
  objectId = '<' || objectId || '>'

  /* WINDOWEDVDM = DOS VDM
	  'SET DPMI_DOS_API=ENABLED;' ||,
	  'SET DPMI_MEMORY_LIMIT=4;' ||,
   */

  setup = 'EXENAME=*;' ||,
	  'PROGTYPE=WINDOWEDVDM;' ||,
	  'SET DOS_HIGH=1;' ||,
	  'SET DOS_UMB=1;' ||,
	  'SET EMS_FRAME_LOCATION=NONE;' ||,
	  'SET EMS_LOW_MAP_REGION=0;' ||,
	  'SET XMS_MEMORY_LIMIT=2048;' ||,
	  'DEFAULTVIEW=RUNNING;' ||,
	  'PARAMETERS=' || parms || ';' ||,
	  'STARTUPDIR=' || dirName || ';' ||,
	  'OBJECTID=' || objectId

  say 'Using objectSeq' objectSeq

  rc = CreateProgramObject(title, locId, setup)
  if rc \= 1 then
    say objectId 'create failed'

  rc = SysOpenObject(objectId, 4, 1)
  if rc \= 1 then
    say objectId 'open failed'

  /* Give object time to open */
  call SysSleep .25

  /* Bring to foreground */
  rc = SysOpenObject(objectId, 4, 1)
  if rc \= 1 then
    say objectId 'open failed'

  call SysFileDelete seqFile
  call lineout seqFile, objectSeq
  call stream seqFile, 'C', 'CLOSE'

  if G.!Keep then
    exit
  else
    'exit'

/*=== end main ===*/

/*=== Initialize: Intialize globals ===*/

Initialize: procedure expose G.

  call GetCmdName
  call LoadRexxUtils
  G.!Env = 'OS2ENVIRONMENT'
  call GetTmpDir

  return

/* end Initialize */

/*=== ScanArgs2(CmdLine) Scan command line ===*/

ScanArgs2: procedure expose G.

  G.!DbgLvl = 0
  G.!Keep = 0
  G.!Verbose = 0
  G.!DirName = ''

  parse arg cmdTail
  cmdTail = strip(cmdTail)

  do while cmdTail \= ''

    parse var cmdTail curArg cmdTail
    cmdTail = strip(cmdTail)

    select
    when curArg == '-d' then
      G.!DbgLvl = G.!DbgLvl + 1
    when curArg == '-h' | curArg == '-?' then
      call ScanArgsHelp
    when curArg == '-k' then
      G.!Keep = 1
    when curArg == '-v' then
      G.!Verbose = 1
    when curArg == '-V' then do
      say G.!CmdName G.!Version
      exit
    end
    when left(curArg, 1) = '-' then
      call ScanArgsUsage curArg 'unexpected'
    otherwise
      if G.!DirName = '' then
	G.!DirName = curArg
      else
	call ScanArgsUsage 'only 1 directory allowed'
    end
  end /* do */

  return

/* end ScanArgs2 */

/*=== ScanArgsHelp() Display ScanArgs usage help exit routine ===*/

ScanArgsHelp:
  say
  say 'skelrexx skeleton REXX script.'
  say
  say 'Usage:' G.!CmdName '[-d] [-h] [-k] [-v] [-V] [-?] directory'
  say
  say '  -d           Enable debug logic, repeat for more verbosity'
  say '  -h -?        Display this message'
  say '  -k           Keep 4OS2 session, default exits'
  say '  -v           Enable verbose output'
  say '  -V           Display version number and quit'
  say
  say '  directory    Directory for shell'
  exit 255

/* end ScanArgsHelp */

/*=== ScanArgsUsage(message) Report Scanargs usage error exit routine ===*/

ScanArgsUsage:
  parse arg msg
  say
  if msg \== '' then
    say msg
  say 'Usage:' G.!CmdName '[-d] [-h] [-k] [-v] [-V] [-?] directory'
  exit 255

/* end ScanArgsUsage */

/*==============================================================================*/
/*=== SkelRexxFunc standards - Delete unused - Move modified above this mark ===*/
/*==============================================================================*/

/*=== CreateProgramObject(name, location, setupString[, opt]) return 1 if OK ===*/

CreateProgramObject: procedure
  name = arg(1)
  location = arg(2)
  setup = arg(3)
  opt = arg(4)
  if opt == '' then
    opt = 'REPLACE'
  /* Create program object */
  say 'Creating' name 'object...'
  rc = SysCreateObject('WPProgram',,
		       name,,
		       location,,
		       setup,,
		       opt)
  return rc				/* 1 if OK */

/* end CreateProgramObject */

/*=== IsFile(fileSpec) return true if arg is file ===*/

IsFile: procedure expose G.
  parse arg fileSpec
  if fileSpec == '' then
    yes = 0
  else do
    call SysFileTree fileSpec, 'fileList', 'F'
    if RESULT \= 0 then
      call Fatal 'IsFile' wildCard 'failed'
    /* Assume caller knows if arg contains wildcards */
    yes = fileList.0 \= 0
  end
  return yes

/* end IsFile */

/*=== MakePath(drv, dir, name, ext) Make pathname from parts ===*/

MakePath: procedure

  /* All parts optional - code guesses what caller means
     MakePath(path, filename) will work
     MakePath(path, name, ext) will not work
     Autmatically converts unix slashes to dos slashes
   */

  parse arg drv, dir, name, ext

  /* If 2 args, assume dir ane name */
  if name == '' & ext == '' then do
    name = dir
    dir = drv
    drv = ''
  end

  if drv == '' then
    path = ''				/* Drive omitted */
  else do
    drv = translate(drv, '\', '/')	/* Ensure DOS */
    path = drv
    /* If just one character assume drive letter */
    if length(drv) = 1 then
      path = path':'
    /* If leading \\ assume UNC */
    else if left(drv, 2) == '\\' & right(drv, 1) \== '\' then
      path = path'\'
    /* If 3 args and drv longer than 2 and no trailing \ on dir assume given dir, name, ext */
    else if ext == '' & right(dir, 1) \= '\' & length(drv) > 2 then do
      ext = name
      name = dir
      dir = ''
      drv = ''
    end
    /* Otherwise figure out later */
  end

  if dir \== '' then do
    dir = translate(dir, '\', '/')	/* Ensure DOS */
    c = right(path, 1)
    c2 = left(dir, 1)
    if path \== '' & c \== ':' & c \== '\' & c2 \== '\' then
      path = path || '\' || dir
    else if c == '\' & c2 == '\' then
      path = path || substr(dir, 2)
    else
      path = path || dir
  end

  if name \== '' then do
    c = right(path, 1)
    if path \== '' & c \= '\' & c \== ':' then
      path = path || '\' || name
    else
      path = path || name
  end

  if ext \== '' then do
    if left(ext, 1) \== '.' then
      path = path'.'ext
    else
      path = path || ext
  end

  return path

/* end MakePath */

/*==========================================================================*/
/*=== SkelRexx standards - Delete unused - Move modified above this mark ===*/
/*==========================================================================*/

/*=== GetTmpDir() Get TMP dir name with trailing backslash, set G. ===*/

GetTmpDir: procedure expose G.
  tmpDir = value('TMP',,G.!Env)
  if tmpDir \= '' & right(tmpDir, 1) \= ':' & right(tmpDir, 1) \== '\' then
    tmpDir = tmpDir'\'			/* Stuff backslash */
  G.!TmpDir = tmpDir
  return

/* end GetTmpDir */

/*=== Error() Report ERROR, FAILURE etc., trace and exit or return if called ===*/

Error:
  say
  parse source . . cmd
  say 'CONDITION'('C') 'signaled at line' SIGL 'of' cmd'.'
  if 'CONDITION'('D') \= '' then say 'REXX reason =' 'CONDITION'('D')'.'
  if 'CONDITION'('C') == 'SYNTAX' & 'SYMBOL'('RC') == 'VAR' then
    say 'REXX error =' RC '-' 'ERRORTEXT'(RC)'.'
  else if 'SYMBOL'('RC') == 'VAR' then
    say 'RC =' RC'.'
  say 'Source =' 'SOURCELINE'(SIGL)

  if 'CONDITION'('I') \== 'CALL' | 'CONDITION'('C') == 'NOVALUE' | 'CONDITION'('C') == 'SYNTAX' then do
    trace '?A'
    say 'Enter REXX commands to debug failure.  Press enter to exit script.'
    call 'SYSSLEEP' 2
    if 'SYMBOL'('RC') == 'VAR' then exit RC; else exit 255
  end

  return

/* end Error */

/*=== GetCmdName() Get script name; set G.!CmdName ===*/

GetCmdName: procedure expose G.
  parse source . . cmd
  cmd = filespec('N', cmd)		/* Chop path */
  c = lastpos('.', cmd)
  if c > 1 then
    cmd = left(cmd, c - 1)		/* Chop extension */
  G.!CmdName = translate(cmd, xrange('a', 'z'), xrange('A', 'Z'))	/* Lowercase */
  return

/* end GetCmdName */

/*=== Fatal(message) Report fatal error and exit ===*/

Fatal:
  parse arg msg
  call 'LINEOUT' 'STDERR', ''
  call 'LINEOUT' 'STDERR', G.!CmdName':' msg 'at script line' SIGL
  call 'BEEP' 200, 300
  call 'SYSSLEEP' 2
  exit 254

/* end Fatal */

/*=== Halt() Report HALT condition and exit ===*/

Halt:
  say
  parse source . . cmd
  say 'CONDITION'('C') 'signaled at' cmd 'line' SIGL'.'
  say 'Source = ' 'SOURCELINE'(SIGL)
  call 'SYSSLEEP' 2
  say 'Exiting.'
  exit 253

/* end Halt */

/*=== LoadRexxUtils: Load fuctions ===*/

LoadRexxUtils:

  /* Add all Rexx functions */
  if RxFuncQuery('SysLoadFuncs') then do
    call RxFuncAdd 'SysLoadFuncs', 'REXXUTIL', 'SysLoadFuncs'
    if RESULT then do
      say 'Cannot load SysLoadFuncs'
      exit
    end
    call SysLoadFuncs
  end /* end do */

  return

/* end LoadRexxUtils */

/* The end */
